<?php
require_once __DIR__ . '/src/bootstrap.php';

$userData = $userObj->getData();
$files = [];

try {
    $files = $r2->listObjects($bucket, $currentUser . '/');
} catch (Exception $e) {
    if (strpos($e->getMessage(), '404') === false) {
        $error = "Failed to list files: " . $e->getMessage();
    }
}

$fileList = [];
foreach ($files as $f) {
    $name = substr($f['key'], strlen($currentUser) + 1);
    if ($name) {
        $fileList[] = [
            'name' => $name,
            'size' => $f['size'],
            'date' => $f['last_modified']
        ];
    }
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Files - TurBay Cloud</title>
    <link href="https://fonts.googleapis.com/css2?family=Outfit:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/style.css">
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        .context-menu {
            display: none;
            position: absolute;
            z-index: 100;
            background: rgba(24, 24, 27, 0.95);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);
            min-width: 180px;
        }

        .menu-item {
            padding: 0.75rem 1rem;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: white;
            transition: background 0.2s;
        }

        .menu-item:hover {
            background: rgba(255, 255, 255, 0.1);
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            z-index: 200;
            align-items: center;
            justify-content: center;
            backdrop-filter: blur(5px);
        }
    </style>
</head>

<body>
    <div class="app-container">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="brand-title">
                <i data-feather="cloud" style="color: var(--accent-primary)"></i> TurBay Cloud
            </div>

            <nav style="flex:1">
                <a href="index.php" class="nav-item active">
                    <i data-feather="hard-drive"></i> Files
                </a>
                <a href="usage.php" class="nav-item">
                    <i data-feather="pie-chart"></i> Usage
                </a>
                <a href="billing.php" class="nav-item">
                    <i data-feather="credit-card"></i> Billing & Packet
                </a>
                <a href="discover.php" class="nav-item">
                    <i data-feather="globe"></i> Discover
                </a>
            </nav>

            <div style="border-top: 1px solid rgba(255,255,255,0.05); padding-top: 1.5rem;">
                <div
                    style="font-size: 0.9rem; font-weight: 500; margin-bottom: 0.5rem; display: flex; justify-content: space-between;">
                    <span>Storage</span>
                    <span class="text-muted"><?= round(($userData['used_storage'] ?? 0) / 1024 / 1024 / 1024, 2) ?>
                        GB</span>
                </div>
                <div class="progress-bar" style="margin-bottom: 1.5rem;">
                    <?php
                    $limit = $userData['storage_limit'] ?? 0;
                    $used = $userData['used_storage'] ?? 0;
                    $pct = ($limit > 0) ? ($used / $limit) * 100 : 0;
                    ?>
                    <div class="progress-fill" style="width: <?= min(100, $pct) ?>%;"></div>
                </div>

                <a href="logout.php" class="nav-item text-muted">
                    <i data-feather="log-out"></i> Logout
                </a>
            </div>
        </div>

        <!-- Main Content -->
        <div class="main-content">
            <div class="page-header">
                <h2>My Files</h2>
                <button onclick="document.getElementById('fileInput').click()"
                    style="width: auto; padding-right: 1.5rem; padding-left: 1.5rem; display: flex; align-items: center; gap: 0.5rem;">
                    <i data-feather="upload-cloud"></i> Upload File
                </button>
                <form action="upload.php" method="POST" enctype="multipart/form-data" id="uploadForm">
                    <input type="file" id="fileInput" name="file" style="display: none;"
                        onchange="document.getElementById('uploadForm').submit()">
                </form>
            </div>

            <?php if (isset($error)): ?>
                <div class="card" style="border-color: var(--danger);">
                    <span style="color: var(--danger)"><?= htmlspecialchars($error) ?></span>
                </div>
            <?php endif; ?>

            <div class="card">
                <?php if (empty($fileList)): ?>
                    <div style="text-align: center; padding: 4rem 0; color: var(--text-secondary);">
                        <i data-feather="inbox" style="width: 48px; height: 48px; margin-bottom: 1rem; opacity: 0.5;"></i>
                        <p>Your cloud is empty. Upload a file to get started.</p>
                    </div>
                <?php else: ?>
                    <table>
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th style="width: 150px;">Size</th>
                                <th style="width: 200px;">Date</th>
                                <th style="width: 120px;">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($fileList as $f): ?>
                                <tr oncontextmenu="showContext(event, '<?= htmlspecialchars($f['name']) ?>')">
                                    <td>
                                        <div style="display: flex; align-items: center; gap: 0.75rem;">
                                            <div
                                                style="background: rgba(59, 130, 246, 0.1); padding: 0.5rem; border-radius: 8px;">
                                                <i data-feather="file"
                                                    style="width: 20px; height: 20px; color: var(--accent-primary)"></i>
                                            </div>
                                            <a href="download.php?file=<?= urlencode($f['name']) ?>"
                                                style="color: white; text-decoration: none; font-weight: 500;">
                                                <?= htmlspecialchars($f['name']) ?>
                                            </a>
                                        </div>
                                    </td>
                                    <td class="text-muted">
                                        <?= round($f['size'] / 1024 / 1024, 2) ?> MB
                                    </td>
                                    <td class="text-muted">
                                        <?= date('M d, Y', strtotime($f['date'])) ?>
                                    </td>
                                    <td>
                                        <div style="display: flex; gap: 0.5rem;">
                                            <!-- Buttons kept for accessibility, but Context Menu handles same actions -->
                                            <button
                                                onclick="selectedFile='<?= htmlspecialchars($f['name']) ?>'; openShareModal()"
                                                class="btn-icon secondary" title="Share">
                                                <i data-feather="share-2" style="width: 16px;"></i>
                                            </button>
                                            <a href="download.php?file=<?= urlencode($f['name']) ?>" class="btn-icon secondary"
                                                title="Download">
                                                <i data-feather="download" style="width: 16px;"></i>
                                            </a>
                                            <form action="delete.php" method="POST"
                                                onsubmit="return confirm('Delete this file?');" style="margin:0;">
                                                <input type="hidden" name="filename"
                                                    value="<?= htmlspecialchars($f['name']) ?>">
                                                <button type="submit" class="btn-icon danger" title="Delete">
                                                    <i data-feather="trash-2" style="width: 16px;"></i>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>

            <div style="margin-top: 1.5rem; opacity: 0.6; font-size: 0.85rem; text-align: center;">
                <i data-feather="info" style="width: 14px; vertical-align: middle;"></i> Tip: Right-click on any file
                for more options.
            </div>

            <!-- Context Menu -->
            <div id="contextMenu" class="context-menu">
                <div class="menu-item" onclick="openShareModal()">
                    <i data-feather="share-2" style="width: 16px;"></i> Share
                </div>
                <div class="menu-item" onclick="triggerDownload()">
                    <i data-feather="download" style="width: 16px;"></i> Download
                </div>
                <div style="height: 1px; background: rgba(255,255,255,0.1); margin: 0.25rem 0;"></div>
                <div class="menu-item" onclick="triggerDelete()" style="color: var(--danger);">
                    <i data-feather="trash-2" style="width: 16px;"></i> Delete
                </div>
            </div>

            <!-- Share Modal -->
            <div id="shareModal" class="modal">
                <div class="card" style="width: 100%; max-width: 450px; animation: slideDown 0.3s;">
                    <h3
                        style="margin-top:0; display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
                        Share File
                        <i data-feather="x" style="cursor: pointer;" onclick="closeShareModal()"></i>
                    </h3>
                    <p id="shareFileName" class="text-muted"
                        style="margin-bottom: 1.5rem; word-break: break-all; font-weight: 500; font-size: 1.1rem; border-bottom: 1px solid rgba(255,255,255,0.1); padding-bottom: 1rem;">
                    </p>

                    <div style="display: flex; gap: 1rem; margin-bottom: 2rem;">
                        <button id="btnShareLink" class="secondary"
                            style="flex: 1; justify-content: center; height: 100px; flex-direction: column; border-width: 2px;"
                            onclick="setShareType('link')">
                            <i data-feather="link" style="width: 24px; height: 24px; margin-bottom: 0.5rem;"></i>
                            Private Link
                            <span
                                style="font-size: 0.75rem; color: var(--text-secondary); margin-top: 0.25rem; font-weight: 400;">Just
                                for those with the link</span>
                        </button>
                        <button id="btnSharePublic" class="secondary"
                            style="flex: 1; justify-content: center; height: 100px; flex-direction: column; border-width: 2px;"
                            onclick="setShareType('public')">
                            <i data-feather="globe" style="width: 24px; height: 24px; margin-bottom: 0.5rem;"></i>
                            Public Forum
                            <span
                                style="font-size: 0.75rem; color: var(--text-secondary); margin-top: 0.25rem; font-weight: 400;">Listed
                                on Discover page</span>
                        </button>
                    </div>

                    <div id="shareResult"
                        style="display: none; background: rgba(0,0,0,0.3); padding: 1rem; border-radius: 8px; margin-bottom: 1rem; border: 1px solid var(--success);">
                        <label
                            style="font-size: 0.8rem; color: var(--success); display: block; margin-bottom: 0.5rem;">Link
                            Created Successfully!</label>
                        <input type="text" id="shareUrl" readonly
                            style="width: 100%; margin-bottom: 0.5rem; background: transparent; border: none; color: white; font-family: monospace; font-size: 0.9rem;">
                        <button onclick="copyShareLink()" class="btn-sm" style="width: 100%;">Copy Link</button>
                    </div>

                    <button id="btnGenerate" onclick="generateShare()" style="width: 100%; padding: 1rem;">Create
                        Link</button>
                </div>
            </div>

        </div>
    </div>
    <script>
        feather.replace();

        let selectedFile = '';
        let currentShareType = 'link';

        // Close context menu on click
        document.addEventListener('click', function (e) {
            document.getElementById('contextMenu').style.display = 'none';
        });

        function showContext(e, fileName) {
            e.preventDefault();
            selectedFile = fileName;
            const menu = document.getElementById('contextMenu');
            menu.style.display = 'block';
            menu.style.left = e.pageX + 'px';
            menu.style.top = e.pageY + 'px';
        }

        function openShareModal() {
            document.getElementById('shareModal').style.display = 'flex';
            document.getElementById('shareFileName').innerText = selectedFile;
            document.getElementById('shareResult').style.display = 'none';
            document.getElementById('btnGenerate').style.display = 'block';
            setShareType('link');
        }

        function closeShareModal() {
            document.getElementById('shareModal').style.display = 'none';
        }

        function setShareType(type) {
            currentShareType = type;
            const btnLink = document.getElementById('btnShareLink');
            const btnPublic = document.getElementById('btnSharePublic');

            // Reset styles
            btnLink.style.borderColor = 'rgba(255,255,255,0.1)';
            btnLink.style.background = 'transparent';
            btnPublic.style.borderColor = 'rgba(255,255,255,0.1)';
            btnPublic.style.background = 'transparent';

            // Active style
            const activeBtn = (type === 'link') ? btnLink : btnPublic;
            activeBtn.style.borderColor = 'var(--accent-primary)';
            activeBtn.style.background = 'rgba(59, 130, 246, 0.1)';
        }

        async function generateShare() {
            const btn = document.getElementById('btnGenerate');
            btn.innerText = 'Creating...';

            try {
                const res = await fetch('api_share_create.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        file_path: '<?= $currentUser ?>/' + selectedFile,
                        type: currentShareType
                    })
                });
                const data = await res.json();

                if (data.success) {
                    document.getElementById('shareResult').style.display = 'block';
                    document.getElementById('shareUrl').value = data.link;
                    btn.style.display = 'none';
                } else {
                    alert('Error: ' + data.error);
                }
            } catch (e) {
                console.error(e);
                alert('An error occurred');
            } finally {
                btn.innerText = 'Create Link';
            }
        }

        function copyShareLink() {
            const copyText = document.getElementById("shareUrl");
            copyText.select();
            document.execCommand("copy");
            alert("Link copied to clipboard!");
        }

        function triggerDownload() {
            window.location.href = 'download.php?file=' + encodeURIComponent(selectedFile);
        }

        function triggerDelete() {
            if (confirm('Are you sure you want to delete ' + selectedFile + '?')) {
                // Post to delete.php via temporary form
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = 'delete.php';
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'filename';
                input.value = selectedFile;
                form.appendChild(input);
                document.body.appendChild(form);
                form.submit();
            }
        }
    </script>
</body>

</html>